﻿<#

  LocalHyper-V Manager v20260220 - Stefan Peters - www.microcloud.nl
  
  Revision history:
  v20260220 - Add [LOCALHYPERVMANAGER] tag in Notes.
            - Only show [LOCALHYPERVMANAGER] VMs in delete selection to prevent dataloss.
            - Fixed 'Enable TPM' on Gen2 VMs.
            - Added error correction on XML file.
            - Added detection for Hyper-V powershell module
            - Added detection for Administrator privileges.
            - Start VM after creation Y/N               
            - Add ISO to new empty VM.
  v202509dv - Delete VM selection change to out-gridview for easier and multi select.
  v20241201 - Made cosmetic improvements for public release
            - Added features:
              -> Create new VM from empty disk
              -> Delete existing VM
              -> Nested Virtualisation
#>

$appversion="v20260220"

# Checks
if (-not (Get-Module -ListAvailable -Name Hyper-V)) {
    Write-Error "Hyper-V module not found. Please install Hyper-V management tools."
    exit 1
}

if (-not (Get-WindowsOptionalFeature -Online -FeatureName Microsoft-Hyper-V-All).State -eq 'Enabled') {
    Write-Error "Hyper-V is not enabled on this system."
    exit 1
}

# Add at script start
if (-not ([Security.Principal.WindowsPrincipal][Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {
    Write-Error "This script requires Administrator privileges."
    exit 1
}
#/Checks
$global:ScriptName           = $MyInvocation.MyCommand.Name                                 # Current script name
$global:ScriptNameAndPath    = $MyInvocation.MyCommand.Definition                           # put full file and pathname in var.
$global:ScriptPath           = $global:ScriptNameAndPath.Replace($global:ScriptName, "")    # current script path
$global:ScriptXMLName        = $global:ScriptName.Replace(".ps1",".xml")                    # XML filename
$global:ScriptXMLNameandPath = $global:ScriptPath+$global:ScriptName.Replace(".ps1",".xml") # XML path+filename

# Read all settings from XML file.
if (-not (Test-Path $global:ScriptXMLNameandPath)) {
    Write-Error "Configuration file not found: $global:ScriptXMLNameandPath"
    exit 1
}

try {
    [xml]$xml = Get-Content $global:ScriptXMLNameandPath -ErrorAction Stop
} catch {
    Write-Error "Failed to parse XML configuration: $_"
    exit 1
}

# Parameters
$global:HyperVPath        = $xml.config.paths.HyperVRoot                 # In this folder the VM is created.
$global:DefaultSwitchName = $xml.config.HyperVSettings.DefaultSwitchName # Default Switch is de the default in Windows 10/11 that allows default internet breakout.
$global:ExtraSwitchName   = $xml.config.HyperVSettings.ExtraSwitchName   # Extra Virtual Switch name if used.

$cpucount=$xml.config.VMSettings.CPUCount  # Default CPU count for a new VM from template
$memory=$xml.config.VMSettings.Memory  # Default RAM for a new VM from template
$defaultOSdisksize=$xml.config.VMSettings.OSDiskSizes.Size # Default OS disk sizes
# Parameters

## Functions
Function CreateVM ($xVMname, $xGen, $xDiffVHDX, $reason, $nic)
  { 
    $timestamp = (Get-Date).ToString("yyyyMMdd-HHmm")
    $xVMpath = Join-Path -Path $global:HyperVPath -ChildPath $xVMname
    $xVMVHDpath = Join-Path -Path $xVMpath -ChildPath "$xVMname-0.vhdx"   
    $notes="[LOCALHYPERVMANAGER]`n"
    $notes+="VM created on: "+$timestamp+"`n"
    $notes+="VM Differential source: "+$xDiffVHDX+"`n"
    $notes+="VM root path: "+$xVMpath+"`n"
    $notes+="Reason: "+$reason
    # NIC
    Switch ($nic)
    {
       1 { $primaryswitch=$global:DefaultSwitchName }
       2 { $primaryswitch=$global:ExtraSwitchName }
       3 { $primaryswitch=$global:DefaultSwitchName
           $secundaryswitch=$global:ExtraSwitchName 
          }
    }  
    # NIC
    $memsize=($memory)
    If ($memsize -like "*GB") 
      {
        $numericMemSize = [int]$memsize.TrimEnd('GB') * 1GB
      } 
    New-Item -Path $xVMpath -ItemType Directory # pre-create target path
    New-VHD -Path $xVMVHDpath -ParentPath $xDiffVHDX -Differencing    
    New-VM -Name $xVMName -MemoryStartupBytes $numericMemSize -VHDPath $xVMVHDpath -Path $xVMpath -Generation $xGen -Switch $primaryswitch
    If ($nic -eq 3)
      {
        Add-VMNetworkAdapter -VMName $xVMName -SwitchName $secundaryswitch
      }
    SET-VMProcessor –VMName $xVMName –count $cpucount   
    Set-VMMemory –VMName $xVMName -DynamicMemoryEnabled $false -StartupBytes $numericMemSize
    Set-VM –VMName $xVMName -AutomaticCheckpointsEnabled $false -Notes $notes        
    If ($xGen -eq 2)
      {
        Write-host -ForegroundColor Yellow "Gen2, enable TPM!"        
	    Set-VMKeyProtector -VMName $xVMName -NewLocalKeyProtector
        Enable-VMTPM -VMName $xVMName        
      }
    Write-host -ForegroundColor Yellow "VM Created !!!" 
    Write-host -ForegroundColor Green "VMName           : $xVMName"
    Write-host -ForegroundColor Green "VM Path          : $xVMpath"
    Write-host -ForegroundColor Green "VM VHD Path      : $xVMVHDpath"
    Write-host -ForegroundColor Green "VHDX Diff Source : $xDiffVHDX"
    Write-host -ForegroundColor Green "Gen              : $xGen"
    Write-host -ForegroundColor Green "Nr of vCPU       : $cpucount"
    Write-host -ForegroundColor Green "RAM              : $($memory/1073741824) GB"
  } #/CreateVM

Function CreateVMEmpty ($xVMname, $xGen, $disksize, $reason, $nic, $isoname)
  { 
    $timestamp = (Get-Date).ToString("yyyyMMdd-HHmm")
    $xVMpath = Join-Path -Path $global:HyperVPath -ChildPath $xVMname
    $xVMVHDpath = Join-Path -Path $xVMpath -ChildPath "$xVMname-0.vhdx"    
    $notes="[LOCALHYPERVMANAGER]`n"
    $notes+="VM created on: "+$timestamp+"`n"    
    $notes+="VM root path: "+$xVMpath+"`n"
    $notes+="Reason: "+$reason     
    # NIC
    Switch ($nic)
    {
       1 { $primaryswitch=$global:DefaultSwitchName }
       2 { $primaryswitch=$global:ExtraSwitchName }
       3 { $primaryswitch=$global:DefaultSwitchName
           $secundaryswitch=$global:ExtraSwitchName 
          }
    }  
    # NIC
    New-Item -Path $xVMpath -ItemType Directory # pre-create target path
    $disksize=($defaultOSdisksize[$disksize])
    If ($disksize -like "*GB") 
      {
        $numericDiskSize = [int]$disksize.TrimEnd('GB') * 1GB
      } 
    New-VHD -Path $xVMVHDpath -SizeBytes $numericDiskSize
    $memsize=($memory)
    If ($memsize -like "*GB") 
      {
        $numericMemSize = [int]$memsize.TrimEnd('GB') * 1GB
      } 
    New-VM -Name $xVMName -MemoryStartupBytes $numericMemSize -VHDPath $xVMVHDpath -Path $xVMpath -Generation $xGen -Switch $primaryswitch
    If ($nic -eq 3)
      {
        Add-VMNetworkAdapter -VMName $xVMName -SwitchName $secundaryswitch
      }
    SET-VMProcessor –VMName $xVMName –count $cpucount
    Set-VMMemory –VMName $xVMName -DynamicMemoryEnabled $false -StartupBytes $numericMemSize
    Set-VM –VMName $xVMName -AutomaticCheckpointsEnabled $false -Notes $notes

    If ($null -ne $ISOname)
      {
        Add-VMDvdDrive -VMName $xVMName -Path $ISOname
        $network = Get-VMNetworkAdapter -VMName $xVMName
        $vhd = Get-VMHardDiskDrive -VMName $xVMName
        $dvd = Get-VMDvdDrive -VMName $xVMName  
        Set-VMFirmware -VMName $xVMName -BootOrder $dvd,$vhd,$network
      }
    If ($xGen -eq 2)
      {
        Write-host -ForegroundColor Yellow "Gen2, enable TPM!"        
	    Set-VMKeyProtector -VMName $xVMName -NewLocalKeyProtector
        Enable-VMTPM -VMName $xVMName        
      }    
    Write-host -ForegroundColor Yellow "VM Created !!!"
    Write-host -ForegroundColor Green "VMName           : $xVMName"
    Write-host -ForegroundColor Green "VM Path          : $xVMpath"
    Write-host -ForegroundColor Green "VM VHD Path      : $xVMVHDpath"    
    Write-host -ForegroundColor Green "Gen              : $xGen"
    Write-host -ForegroundColor Green "Nr of vCPU       : $cpucount"
    Write-host -ForegroundColor Green "RAM              : $($memory/1073741824) GB"
  } #/CreateVMEmpty

Function GoCreateVMTemplate
  {     
    $list=Get-Childitem $($xml.config.paths.Templates.folder)
    Write-host -ForegroundColor Yellow "Choose template:"
    $vmcounter = 1
    Foreach ($l in $list)
        {
          Write-host "  $($vmcounter) - $($l.name)"
          $vmcounter++
        }
    Write-host "  Q - Quit"
    $choice1 = read-host "Enter the number of the template"
    If ($choice1 -ne "q")
      {
        $choice1=$choice1-1
        Write-host
        $xtemplatename=$($list[$choice1].fullname)    
        If ($xtemplatename.Contains("gen2"))
            {
            $xGen=2
            } else { $xGen=1 }   
        Write-host "Template selected: $($list[$choice1].fullname) - Generation: $xGen"
        Write-host
        Write-host -ForegroundColor Yellow "Connect new vm to Hyper-V switch:"
        Write-host "1 - $global:DefaultSwitchName"
        Write-host "2 - $global:ExtraSwitchName"
        Write-host "3 - $($global:DefaultSwitchName) and $($global:ExtraSwitchName)"
        Write-host "Q - Quit"
        Write-host
        $choice2 = read-host "Enter the number of your choice"
        Write-host
        Switch ($choice2)
        {
            1 { $nic=1 }
            2 { $nic=2 }
            3 { $nic=3 }            
            Q { }
            Default { $nic=1 }
        }  
        If ($choice2 -ne "q")
          {
            Write-Host "HV Path: $global:hypervpath"
            $vmname = read-host "VMName"
            $reason = $null
            $reason = read-host "Enter the reason for this VM?"
            CreateVM $vmname $xGen $($list[$choice1].fullname) $reason $nic $true            
            $choice3 = read-host "Do you want to start the new VM (Y/N):"            
            Switch ($choice3.ToUpper())
              {
                Y { Start-VM -Name $vmname }
                N { Write-host "No" }
              }
          }
      }
  } #/GoCreateVMTemplate

Function GoCreateVMEmpty
  {    
    Write-host -ForegroundColor Yellow "Create new VM with empty disk, first choose the Generation"
    Write-host "1 - Generation 1 (Legacy OS)"
    Write-host "2 - Generation 2 (Required for modern Windows 11 with TPM)"
    Write-host "Q - Quit"
    $choice1 = read-host "Enter the number of the Hyper-V VM - Generation"    
    If ($choice1 -ne "q")
      {
        Switch ($choice1)
        {
          1 { $xGen=1 }
          2 { $xGen=2 }      
          Default { $xGen=2 }
        }
        #$defaultOSdisksize
        Write-host
        Write-host -ForegroundColor Yellow "Next we need to know the OS disk size:"
        $vmcounter = 1
        Foreach ($disksize in $defaultOSdisksize)
            {
              Write-host "  $($vmcounter) - $($($disksize)/1073741824) GB"
              $vmcounter++
            }
        Write-host "  Q - Quit"
        $choice1 = read-host "Enter the disksize for the empty OS disk"
        If ($choice1 -ne "q")
          {
            $choice1=$choice1-1
            $disksize=$choice1
            Write-host
            Write-host -ForegroundColor Yellow "Connect new vm to Hyper-V switch:"
            Write-host "1 - $global:DefaultSwitchName"
            Write-host "2 - $global:ExtraSwitchName"
            Write-host "3 - $($global:DefaultSwitchName) and $($global:ExtraSwitchName)"
            Write-host "Q - Quit"
            Write-host
            $choice2 = read-host "Enter the number of your choice"
            If ($choice2 -ne "q")
              {
                Write-host
                Switch ($choice2)
                {
                   1 { $nic=1 }
                   2 { $nic=2 }
                   3 { $nic=3 }
                   Default { $nic=1 }
                }
                Write-Host "HV Path: $global:hypervpath"                
                $list=Get-Childitem $($xml.config.paths.ISOs.folder)
                Write-host -ForegroundColor Yellow "Choose ISO:"
                $vmcounter = 1
                Foreach ($l in $list)
                    {
                      Write-host "  $($vmcounter) - $($l.name)"
                      $vmcounter++
                    }
                Write-host "  N - NO ISO"
                Write-host "  Q - Quit"
                $choice3 = read-host "Enter the number of the ISO"
                If ($choice3.ToLower() -ne "q" -and $choice3.ToLower() -ne "n")
                  {
                    read-host "check1"
                    $choice3=$choice3-1
                    Write-host
                    $xISOname=$($list[$choice3].fullname)
                  } else { $xISOname=$null }                
                If ($choice3.ToLower() -ne "q")
                  {
                    $vmname = read-host "VMName"
                    $reason = $null
                    $reason = read-host "Enter the reason for this VM?"
                    CreateVMEmpty $vmname $xGen $disksize $reason $nic $xISOname             

                    $choice4 = read-host "Do you want to start the new VM (Y/N)"            
                    Switch ($choice4.ToUpper())
                      {
                        Y { Start-VM -Name $vmname }
                        N { Write-host "No" }
                      }
                  }
              }
          }
      }  
  } #/GoCreateVMEmpty

Function GoDeleteVM
  {
    $list=@()
    $templist=Get-VM 
    Foreach ($item in $templist)
      {
        $notes=$($item.notes)        
        If ( ($($notes) -notlike "*PRODUCTION*") -and ($($notes) -like "*LOCALHYPERVMANAGER*") )
          { 
            $list+=$item           
          } 
      }
    $selection=$list | out-gridview -Title "Choose the VM or VMs you want to delete:" -OutputMode Multiple
    $global:sel=$selection
    If ($null -eq $selection)
      {
        Write-host "Nothing selected."
        Start-sleep 1
        return
      } else
      {
         # Find the longest name
         $longest=0
         Foreach ($sel in $selection)
           {
             If ($longest -le $($sel.name).length) { $longest=$($sel.name).length }
           }
         Write-host -ForegroundColor Yellow "Are you sure you want to delete this/these VM(s): " 

         $DisplayName="VM Name".padright($longest)
         $DisplayPath="Path to VM"
         Write-host -ForegroundColor green "$DisplayName -> $DisplayPath"

         
         Foreach ($sel in $selection)
           {
             $DisplayName=$($sel.name).padright($longest)
             $DisplayPath=Split-Path $($sel.path)
             Write-host -ForegroundColor red "$DisplayName -> Path: $DisplayPath"                      
           }
         Write-host -ForegroundColor Yellow " (YES/NO)?: " -NoNewline         
         $choice2 = (read-host).ToUpper()
         If ($choice2 -eq 'YES')
          {
            Write-Host
            Foreach ($sel in $selection)
              {
                $VMRootPath=Split-Path $($sel.path)
                $renamedpath=$($VMRootPath)+".DELETEME"
                remove-vm -Name $($sel.name) -Force
                Rename-Item -Path $VMRootPath -NewName $renamedpath
                Write-host -ForegroundColor Green "VM [$($sel.name)] has been removed, and the folder [$($sel.path)] has been renamed to: [$renamedpath], feel free to delete it at your convenience"
              } 
            Write-Host         
          } else { Write-host "No!" }
          Read-Host "Press enter to continue"       
      }   
  } #/GoDeleteVM

Function NestedV
  {    
    $templist=Get-VM 
    $vmcounter=1
    Write-host -ForegroundColor Yellow "Choose VM to check Nested Virtualisation:"
    Write-host 
    Write-host "Count   VMName                 Nested virtualisation"
    Write-host "----------------------------------------------------"
    Foreach ($item in $templist)
      {
        $VMP=Get-VMProcessor -vmname $($item.name)
        $space=" " * (20-$($item.name).Length)
        $vmname=$($item.name)+$space
        $space=" " * (5-$($vmcounter).Length)
        $vmc=$space+$vmcounter
        If ($true -eq $($VMP.ExposeVirtualizationExtensions))
          {
            Write-host -ForegroundColor red   "$($vmc)   $($vmname)   $($VMP.ExposeVirtualizationExtensions)"
          } else
          {
            Write-host -ForegroundColor green "$($vmc)   $($vmname)   $($VMP.ExposeVirtualizationExtensions)"
          }
        $vmcounter++
      } 
    Write-host "    Q - Quit"       
    Write-host
    $choice1 = read-host "Enter the number of the VM"
    If ($choice1 -eq "Q")
      {
      } else
      {
        $choice1=$choice1-1
        Write-host
        Write-host "Change the Nested Virtualisation setting for VM: $($templist[$choice1].name) from " -nonewline
        $VMP=Get-VMProcessor -vmname $($templist[$choice1].name)
        If ($true -eq $($VMP.ExposeVirtualizationExtensions))
              { 
                $EVEsetting=$false
                Write-host -ForegroundColor red $($VMP.ExposeVirtualizationExtensions) -NoNewline
                Write-host " to " -NoNewline
                Write-host -ForegroundColor green $($EVEsetting) -NoNewline
                
              } else 
              {
                $EVEsetting=$true
                Write-host -ForegroundColor green $($VMP.ExposeVirtualizationExtensions) -NoNewline
                Write-host " to " -NoNewline
                Write-host -ForegroundColor red $($EVEsetting) -NoNewline
              } 
        Write-host " (Y/N): " -NoNewline        
        $choice2 = (read-host).ToUpper()
        If ($choice2 -eq "Y")
          { 
            Set-VMProcessor -vmname $($templist[$choice1].name) -ExposeVirtualizationExtensions $EVEsetting
            Write-host "Done, changed setting from: $(!$EVEsetting) to: $($EVEsetting)"
            start-sleep 2
          } else
          {
            Write-host "No, nothing changed."
            start-sleep 2
          }
      }
  } #/NestedV

Function ListVMs
  {
    $vmList=@()
    $global:vmList=@()
    $LocalHyperVList=Get-VM     
    Foreach ($item in $LocalHyperVList)
      {
        $VMP=Get-VMProcessor -vmname $($item.name)
        #$($VMP.ExposeVirtualizationExtensions)
        $vmItem = [ordered]@{}
        $vmItem.add(('name'), $($item.name)) #
        $vmItem.add(('state'), $($item.state)) #
        $vmItem.add(('generation'), $($item.generation)) #
        $vmItem.add(('NestedVirtualization'), $($VMP.ExposeVirtualizationExtensions)) #
        $notes=$($item.notes) -replace "`n", "/"
        $vmItem.add(('Notes'), $($notes)) #
        $vmList+=New-Object PSObject -property $vmItem 
      }
    $global:vmList=$vmList
    Write-host -ForegroundColor Yellow "All local Hyper-V machines"    
    $vmlist | out-gridview -title "All local Hyper-V machines"
    Write-host
    read-host "Press enter to go back to the menu <ENTER>"
   
  } #/ListVMs

Function FirstMenu
  {
    Do
    {  
      Write-host
      Write-Host "_    ____ ____ ____ _       _  _ _   _ ___  ____ ____    _  _    _  _ ____ _  _ ____ ____ ____ ____ "
      Write-Host "|    |  | |    |__| |       |__|  \_/  |__] |___ |__/ __ |  |    |\/| |__| |\ | |__| | __ |___ |__/ "
      Write-Host "|___ |__| |___ |  | |___    |  |   |   |    |___ |  \     \/     |  | |  | | \| |  | |__] |___ |  \ "
      Write-host
      Write-host $appversion" - https://www.microcloud.nl"
      Write-host
      Write-host "1 - Create new VM from template"
      Write-host "2 - Create new VM with empty disk"
      Write-host "3 - Delete existing VM"
      Write-host "4 - Nested Virtualisation"    
      Write-host "9 - VM Overview"
      Write-host "Q - Quit"
      Write-host
      $choice = read-host "What is your choice"
      Write-host
      Switch ($choice)
        {
          1 { GoCreateVMTemplate }
          2 { GoCreateVMEmpty }
          3 { GoDeleteVM }
          4 { NestedV }          
          9 { ListVMs }     
        }
    } While ($choice -ne "Q")
 }

FirstMenu